﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Data.Linq;
using VeteransAffairs.Registries.BusinessManager.Utilities;
using VeteransAffairs.Registries.BusinessAHOBPR;
using System.Reflection;

namespace VeteransAffairs.Registries.BusinessManagerAHOBPR
{
    public class AHOBPRRegistrantManager:AHOBPRBaseBO, IRegistrantManager
    {
        private AHOBPRShared _sharedManager = new AHOBPRShared();
  
        #region Methods for GetRegistrantById

        /// <summary>
        /// Select Registrant by edipi or ssn
        /// </summary>
        /// <param name="edipi"></param>
        /// <param name="ssn"></param>
        /// <returns></returns>
        public REGISTRANT GetRegistrantById(string edipi, string ssn)
        {
            REGISTRANT registrant = null;
            using (_dbAhobpr = GetDataContext())
            {
                _dbAhobpr.DeferredLoadingEnabled = false;
                 registrant = (from e in _dbAhobpr.REGISTRANTs 
                              where e.EDIPI == edipi || e.Snum == ssn
                              select e).FirstOrDefault();

            }
            return registrant;
        }

        /// <summary>
        /// Get registrant by user ID
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        public REGISTRANT GetRegistrantByUserId(string userId)
        {
            REGISTRANT registrant = null;
            using (_dbAhobpr = GetDataContext())
            {
                _dbAhobpr.DeferredLoadingEnabled = false;
                registrant = (from e in _dbAhobpr.REGISTRANTs
                              where e.USER_ID == userId
                              select e).FirstOrDefault();

            }
            return registrant;
        }

        /// <summary>
        /// Create new object if registrant is null or update DB fields if it exists
        /// </summary>
        /// <param name="registrant"></param>
        public REGISTRANT InitializeRegistrantDBObject(REGISTRANT registrant)
        {
            if (registrant == null)
            {
                registrant = new REGISTRANT();
                registrant.CREATED = DateTime.Now;
                registrant.UPDATED = DateTime.Now;
                registrant.SetAsChangeTrackingRoot();
                registrant.SetAsInsertOnSubmit();
            }
            else
            {
                registrant.UPDATED = DateTime.Now;
                registrant.UPDATEDBY = "REST Service";
                registrant.SetAsChangeTrackingRoot();
                registrant.SetAsUpdateOnSubmit();
            }
            return registrant;
        }

        public REGISTRANT InitializeRegistrantDBObject(string userId)
        {
            REGISTRANT registrant = GetRegistrantByUserId(userId);
            return InitializeRegistrantDBObject(registrant);
        }

        public REGISTRANT InitializeRegistrantDBObject(string edipi, string ssn)
        {
            REGISTRANT registrant = GetRegistrantById(edipi, ssn);
            return InitializeRegistrantDBObject(registrant);
        }

        /// <summary>
        /// Get total number of participants
        /// </summary>
        /// <returns></returns>
        public string GetTotalParticipants()
        {
            int totalParticipants = 0;
            using (_dbAhobpr = GetDataContext())
            {
                totalParticipants = (from e in _dbAhobpr.REGISTRANTs
                                     where e.STD_REGISTRANT_STATUS_ID == (int)Enums.RegistrantStatus.Participant
                                     select e.REGISTRANT_ID).Count();

            }
            return totalParticipants.ToString("N0");
        }


        /// <summary>
        /// Get userId by Registrant ID
        /// </summary>
        /// <param name="registrantId"></param>
        /// <returns></returns>
        public string GetUserIdByRegistrantId(int registrantId)
        {
            string userId = string.Empty;
            using (_dbAhobpr = GetDataContext())
            {
                _dbAhobpr.DeferredLoadingEnabled = false;
                userId = (from e in _dbAhobpr.REGISTRANTs
                              where e.REGISTRANT_ID == registrantId
                              select e.USER_ID).FirstOrDefault();

            }
            return userId;
        }
        #endregion

        #region Methods for GetRegistrantByUserIdWithDetails

        /// <summary>
        /// Select Registrant by User ID
        /// </summary>
        /// <param name="userId"></param>
          /// <returns></returns>
        public REGISTRANT GetRegistrantByUserIdWithInfo(string userId)
        {
            REGISTRANT registrant = null;
            if (!string.IsNullOrEmpty(userId))
            {
                using (_dbAhobpr = GetDataContext())
                {
                    SetLoadWithForRegistrantInfo(_dbAhobpr);
                    registrant = (from e in _dbAhobpr.REGISTRANTs
                                  where e.USER_ID == userId
                                  select e).FirstOrDefault();

                }
            }
            return registrant;
        }

        public REGISTRANT GetRegistrantByUserIdWithInfoDeploymentOnly(string userId)
        {
            REGISTRANT registrant = null;
            if (!string.IsNullOrEmpty(userId))
            {
                using (_dbAhobpr = GetDataContext())
                {
                    SetLoadWithForRegistrantInfoDeploymentOnly(_dbAhobpr);
                    registrant = (from e in _dbAhobpr.REGISTRANTs
                                  where e.USER_ID == userId
                                  select e).FirstOrDefault();

                }
            }
            return registrant;
        }

        private void SetLoadWithForRegistrantInfoDeploymentOnly(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<REGISTRANT>(e => e.REGISTRANT_DEPLOYMENTs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        private void SetLoadWithForRegistrantInfo(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<REGISTRANT>(e => e.REGISTRANT_CONTACT_INFO);
            lo.LoadWith<REGISTRANT>(e => e.REGISTRANT_RECORD_INFO);
            lo.LoadWith<REGISTRANT_CONTACT_INFO>(e => e.REGISTRANT_ADDRESSes);
            lo.LoadWith<REGISTRANT_CONTACT_INFO>(e => e.REGISTRANT_EMAILs);
            lo.LoadWith<REGISTRANT_CONTACT_INFO>(e => e.REGISTRANT_PHONEs);
            lo.LoadWith<REGISTRANT>(e => e.REGISTRANT_DEPLOYMENTs);
            lo.LoadWith<REGISTRANT>(e => e.REGISTRANT_SERVICE_EPISODEs);
            lo.LoadWith<REGISTRANT_DEPLOYMENT>(e => e.FORM_RESPONSE_QUESTIONs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        /// <summary>
        /// Select Registrant by User ID
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        public REGISTRANT GetRegistrantByUserIdWithResponse(string userId)
        {
            REGISTRANT registrant = null;
            if (!string.IsNullOrEmpty(userId))
            {
                using (_dbAhobpr = GetDataContext())
                {
                    SetLoadWithForRegistrantResponse(_dbAhobpr);
                    registrant = (from e in _dbAhobpr.REGISTRANTs
                                  where e.USER_ID == userId
                                  select e).FirstOrDefault();

                }
            }
            return registrant;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        public REGISTRANT GetRegistrantByUserIdWithFollowupResponse(string userId)
        {
            REGISTRANT registrant = null;
            if (!string.IsNullOrEmpty(userId))
            {
                using (_dbAhobpr = GetDataContext())
                {
                    SetLoadWithForRegistrantFollowupResponse(_dbAhobpr);
                    registrant = (from e in _dbAhobpr.REGISTRANTs
                                  where e.USER_ID == userId
                                  select e).FirstOrDefault();

                }
            }
            return registrant;
        }

        private void SetLoadWithForRegistrantFollowupResponse(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<REGISTRANT>(e => e.FOLLOWUP_FORM_RESPONSEs);
            lo.LoadWith<FOLLOWUP_FORM_RESPONSE>(e => e.FOLLOWUP_FORM_RESPONSE_QUESTIONs);
            lo.LoadWith<FOLLOWUP_FORM_RESPONSE>(e => e.STD_FORM);
            lo.LoadWith<FOLLOWUP_FORM_RESPONSE_QUESTION>(e => e.FOLLOWUP_FORM_RESPONSE_ANSWERs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        private void SetLoadWithForRegistrantResponse(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<REGISTRANT>(e => e.FORM_RESPONSEs);
            lo.LoadWith<FORM_RESPONSE>(e => e.FORM_RESPONSE_STATUS);
            lo.LoadWith<FORM_RESPONSE>(e => e.FORM_RESPONSE_QUESTIONs);
            lo.LoadWith<FORM_RESPONSE_QUESTION>(e => e.FORM_RESPONSE_ANSWERs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        #endregion

        #region Save Registrant
        public int UpdateRegistrant(REGISTRANT registrant)
        {
            return UpdateRegistrant(registrant, 1);
        }

        /// <summary>
        /// Update 
        /// </summary>
        /// <param name="registrant"></param>
        /// <returns></returns>
        private int UpdateRegistrant(REGISTRANT registrant, int tryCount)
        {
            if (tryCount >= 20)
            {
                _sharedManager.LogErrorMessage("Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name,
                    "Unable to save JSON Data after 20 tries: {UserId:" + registrant.USER_ID + "}");

                return -1;
            }
            if (tryCount > 1)
            {
                _sharedManager.LogErrorMessage("Retry", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name,
                    "retry #" + tryCount.ToString() + ": {UserId:" + registrant.USER_ID + "}");
            }

            int returnStatus = 0;

            if (registrant != null)
            {
                using (_dbAhobpr = GetDataContext())
                {
                    _dbAhobpr.DeferredLoadingEnabled = false;

                    //this line traverses all entities, attaching all of them as appropriate to the data context.
                    registrant.SynchroniseWithDataContext(_dbAhobpr);

                    //Check if any actual changes will occur
                    ChangeSet changeSet = _dbAhobpr.GetChangeSet();

                    if (changeSet.Updates.Count > 0 || changeSet.Inserts.Count > 0)
                    {
                        //if changes present then submit changes
                        try
                        {
                            _dbAhobpr.SubmitChanges(ConflictMode.ContinueOnConflict);
                            returnStatus = 1;

                        }
                        catch (System.Data.SqlClient.SqlException ex)
                        {
                            _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, "UserId: " + registrant.USER_ID + " Error: " + ex.Message);
                            returnStatus = UpdateRegistrant(registrant, tryCount + 1);
                         }
                        catch (ChangeConflictException ex)
                        {
                            _dbAhobpr.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                            _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, "UserId: " + registrant.USER_ID + " Error: " + ex.Message);

                            returnStatus = UpdateRegistrant(registrant, tryCount + 1);
                        }
                        catch (Exception ex)
                        {
                            _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, "UserId: " + registrant.USER_ID + " Error: " + ex.Message);
                            returnStatus = UpdateRegistrant(registrant, tryCount + 1);
                        }
                    }
                }
            }
            return returnStatus;
        }
        #endregion

        #region Methods for Flags
        /// <summary>
        /// Add admin flag by flag name
        /// </summary>
        /// <param name="registrantId"></param>
        /// <param name="registryFlagName"></param>
        /// <returns></returns>
        public int AddFlagToRegistrantByFlagName(int registrantId, string registryFlagName, string createdBy)
        {
            int result = 0;

            int registryFlagId = GetRegistryFlagIdByName(registryFlagName);
            if (registryFlagId == 0)
            {
                AddRegistryFlag((int)Enums.AhobprRegistryFlagType.AdminFlag, registryFlagName, registryFlagName, createdBy);
                registryFlagId = GetRegistryFlagIdByName(registryFlagName);
            }
            if (registryFlagId > 0)
            {
                result = AddFlagToRegistrantByFlagId(registrantId, registryFlagId, createdBy);
            }

            return result;
        }

        /// <summary>
        ///  Check if a subpopulation flag already exists
        /// </summary>
        /// <param name="registrantId"></param>
        /// <param name="registryFlagId"></param>
        /// <returns></returns>
       public bool HasSubpopulationFlag(int registrantId, int registryFlagId)
        {
            bool hasSubpopulationFlag = false;

            using (_dbAhobpr = GetDataContext())
            {
                hasSubpopulationFlag = (from e in _dbAhobpr.REGISTRANT_REGISTRY_FLAGs
                              where e.REGISTRANT_ID == registrantId
                                  && e.STD_REGISTRY_FLAG_ID == registryFlagId
                                  && e.STD_REGISTRY_FLAG.STD_REGISTRY_FLAG_TYPE_ID == (int)Enums.AhobprRegistryFlagType.SubpopulationFlag
                              select e).ToList().Count > 0;
            }

            return hasSubpopulationFlag;
        }

        /// <summary>
        /// Add admin flag by flag ID
        /// </summary>
        /// <param name="registrantId"></param>
        /// <param name="registryFlagId"></param>
        /// <returns></returns>
        public int AddFlagToRegistrantByFlagId(int registrantId, int registryFlagId, string createdBy)
        {
            int returnStatus = 0;

            if (registryFlagId == 0 || registrantId == 0)
            {
                return returnStatus;
            }

            using (_dbAhobpr = GetDataContext())
            {
                _dbAhobpr.DeferredLoadingEnabled = false;

                REGISTRANT_REGISTRY_FLAG registrantflag = new REGISTRANT_REGISTRY_FLAG();

                registrantflag.REGISTRANT_ID = registrantId;
                registrantflag.STD_REGISTRY_FLAG_ID = registryFlagId;
                registrantflag.CREATEDBY = createdBy;
                registrantflag.UPDATEDBY = createdBy;

                _dbAhobpr.REGISTRANT_REGISTRY_FLAGs.InsertOnSubmit(registrantflag);

                try
                {
                    _dbAhobpr.SubmitChanges(ConflictMode.ContinueOnConflict);

                    returnStatus = 1;
                }
                catch (System.Data.SqlClient.SqlException ex)
                {
                    _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                    returnStatus = -1;
                }
                catch (ChangeConflictException)
                {
                    _dbAhobpr.ChangeConflicts.ResolveAll(RefreshMode.KeepCurrentValues);

                    try
                    {
                        _dbAhobpr.SubmitChanges(ConflictMode.FailOnFirstConflict);

                        returnStatus = 1;
                    }
                    catch (Exception ex)
                    {
                        _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                        returnStatus = -1;
                    }
                }
                catch
                {
                    returnStatus = 0;
                }
            }
            return returnStatus;
        }

        /// <summary>
        /// Get flag ID by name
        /// </summary>
        /// <param name="registryFlagName"></param>
        /// <returns></returns>
        public int GetRegistryFlagIdByName(string registryFlagName)
        {
            int registryFlagId = 0;

            using (_dbAhobpr = GetDataContext())
            {
                _dbAhobpr.DeferredLoadingEnabled = false;
                registryFlagId = (from e in _dbAhobpr.STD_REGISTRY_FLAGs
                                  where e.REGISTRY_FLAG_NAME == registryFlagName
                                  select e.STD_REGISTRY_FLAG_ID).FirstOrDefault();

            }

            return registryFlagId;
        }


        /// <summary>
        /// Add a new registry flag
        /// </summary>
        /// <param name="registryFlagName"></param>
        /// <param name="registryFlagDescription"></param>
        /// <returns></returns>
        public int AddRegistryFlag(int registryFlagTypeId, string registryFlagName, string registryFlagDescription, string createdBy)
        {
            int returnStatus = 0;

            if (string.IsNullOrEmpty(registryFlagName))
            {
                return returnStatus;
            }

            using (_dbAhobpr = GetDataContext())
            {
                _dbAhobpr.DeferredLoadingEnabled = false;

                //check if registry flag exists
                STD_REGISTRY_FLAG registryFlag = (from e in _dbAhobpr.STD_REGISTRY_FLAGs
                                                  where e.REGISTRY_FLAG_NAME == registryFlagName
                                                  select e).FirstOrDefault();

                //insert if not exists
                if (registryFlag == null)
                {
                    registryFlag = new STD_REGISTRY_FLAG();

                    registryFlag.STD_REGISTRY_FLAG_TYPE_ID = registryFlagTypeId;
                    registryFlag.REGISTRY_FLAG_NAME = registryFlagName;
                    registryFlag.REGISTRY_FLAG_DESCRIPTION = registryFlagDescription;
                    //set active_Flag to false so the admin flags don't show up in the dropdownlist for automatic generated sent messages
                    registryFlag.ACTIVE_FLAG = true;
                    registryFlag.PREDEFINED = true;
                    registryFlag.CREATEDBY = createdBy;
                    registryFlag.UPDATEDBY = createdBy;

                    _dbAhobpr.STD_REGISTRY_FLAGs.InsertOnSubmit(registryFlag);

                    try
                    {
                        _dbAhobpr.SubmitChanges(ConflictMode.ContinueOnConflict);

                        returnStatus = 1;
                    }
                    catch (System.Data.SqlClient.SqlException ex)
                    {
                        _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                        returnStatus = -1;
                    }
                    catch (ChangeConflictException)
                    {
                        _dbAhobpr.ChangeConflicts.ResolveAll(RefreshMode.KeepCurrentValues);

                        try
                        {
                            _dbAhobpr.SubmitChanges(ConflictMode.FailOnFirstConflict);
                            returnStatus = 1;
                        }
                        catch (Exception ex)
                        {
                            _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                            returnStatus = -1;
                        }
                    }
                    catch
                    {
                        returnStatus = 0;
                    }

                }
                // if the registry flag already exists, set returnStatus to 2
                else
                {
                    returnStatus = 2;
                }
            }
            return returnStatus;
        }

        #endregion

        #region Methods for SelectRegistrantNameById

        public string SelectRegistrantNameById(int registrantId)
        {
            string registrant = string.Empty;

            using (_dbAhobpr = GetDataContext())
            {
                registrant = (from e in _dbAhobpr.REGISTRANTs
                              where e.REGISTRANT_ID == registrantId
                              select e.LAST_NAME + ", " + e.FIRST_NAME + " " + e.MIDDLE_NAME).FirstOrDefault();

            }
            return registrant;
        }

        #endregion

        #region Methods for questionnaire related
        /// <summary>
        /// 
        /// </summary>
        /// <param name="registrantId"></param>
        /// <param name="followupFormId"></param>
        /// <returns></returns>
        public int AddFollowupQuestionsToOneReigstrant(int registrantId, int followupFormId, ref bool exists)
        {
            int returnStatus = 0;
            if (registrantId == 0 || followupFormId == 0)
            {
                return returnStatus;
            }

            using (_dbAhobpr = GetDataContext())
            {
                _dbAhobpr.DeferredLoadingEnabled = false;

                //check if registry flag exists
                FOLLOWUP_FORM_RESPONSE followupFormResponse = (from e in _dbAhobpr.FOLLOWUP_FORM_RESPONSEs
                                                               where e.REGISTRANT_ID == registrantId && e.STD_FORM_ID == followupFormId
                                                               select e).FirstOrDefault();

                //insert if not exists
                if (followupFormResponse == null)
                {
                    exists = false;
                    followupFormResponse = new FOLLOWUP_FORM_RESPONSE();

                    followupFormResponse.REGISTRANT_ID = registrantId;
                    followupFormResponse.STD_FORM_ID = followupFormId;

                    _dbAhobpr.FOLLOWUP_FORM_RESPONSEs.InsertOnSubmit(followupFormResponse);

                    try
                    {
                        _dbAhobpr.SubmitChanges(ConflictMode.ContinueOnConflict);

                        returnStatus = 1;
                    }
                    catch (System.Data.SqlClient.SqlException ex)
                    {
                        _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                        returnStatus = -1;
                    }
                    catch (ChangeConflictException)
                    {
                        _dbAhobpr.ChangeConflicts.ResolveAll(RefreshMode.KeepCurrentValues);

                        try
                        {
                            _dbAhobpr.SubmitChanges(ConflictMode.FailOnFirstConflict);

                            returnStatus = 1;
                        }
                        catch (Exception ex)
                        {
                            _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                            returnStatus = -1;
                        }
                    }
                    catch
                    {
                        returnStatus = 0;
                    }

                }
                // if the registry flag already exists, ignore and consider as success insert
                else
                {
                    exists = true;
                    returnStatus = 1;
                }
            }

            return returnStatus;
        }
        #endregion
    }
}
